"use client";

import { useEffect, useMemo, useRef, useState } from "react";
import { motion, AnimatePresence } from "framer-motion";

function fmtDate(iso) {
  try {
    const d = new Date(iso);
    return new Intl.DateTimeFormat("sq-AL", {
      weekday: "long",
      year: "numeric",
      month: "long",
      day: "2-digit",
      hour: "2-digit",
      minute: "2-digit",
    }).format(d);
  } catch {
    return iso;
  }
}

export default function InvitationClient({ token, initial }) {
  const [data, setData] = useState(() => ({
    event: initial.event,
    guest: initial.guest,
    rsvp: initial.rsvp,
  }));

  const [opened, setOpened] = useState(false);
  const [muted, setMuted] = useState(false);
  const [saving, setSaving] = useState(false);
  const audioRef = useRef(null);

  const inviteTitle = useMemo(() => {
    const a = data.event.coupleA || "";
    const b = data.event.coupleB || "";
    return `${a} & ${b}`.trim();
  }, [data.event.coupleA, data.event.coupleB]);

  useEffect(() => {
    // preload audio
    audioRef.current = new Audio("/audio/intro.wav");
    audioRef.current.loop = true;
    audioRef.current.volume = 0.7;
    return () => {
      try { audioRef.current?.pause(); } catch {}
    };
  }, []);

  useEffect(() => {
    if (!audioRef.current) return;
    audioRef.current.muted = muted;
  }, [muted]);

  async function openInvite() {
    setOpened(true);
    try {
      await audioRef.current?.play();
    } catch {
      // Autoplay may fail; user can press play button later.
    }
  }

  async function refresh() {
    const res = await fetch(`/api/invites/${token}`);
    if (!res.ok) return;
    const json = await res.json();
    setData({ event: json.event, guest: json.guest, rsvp: json.rsvp });
  }

  async function submitRsvp(status) {
    setSaving(true);
    try {
      const count = Number(document.getElementById("rsvpCount")?.value || "1");
      const note = String(document.getElementById("rsvpNote")?.value || "");
      const res = await fetch(`/api/invites/${token}/rsvp`, {
        method: "POST",
        headers: { "content-type": "application/json" },
        body: JSON.stringify({ status, count, note }),
      });
      if (res.ok) await refresh();
    } finally {
      setSaving(false);
    }
  }

  return (
    <main className="min-h-screen bg-neutral-950 text-white">
      <AnimatePresence>
        {!opened && (
          <motion.div
            className="fixed inset-0 z-50 grid place-items-center bg-gradient-to-b from-black to-neutral-950"
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
          >
            <div className="mx-auto max-w-lg px-6 text-center">
              <div className="text-xs tracking-widest text-neutral-300">FTESA DIGJITALE</div>
              <div className="mt-3 text-4xl font-semibold tracking-tight">{inviteTitle}</div>
              <div className="mt-2 text-neutral-300">Ju ftojne ne dasmen e tyre</div>

              <div className="mt-8 rounded-2xl border border-white/10 bg-white/5 p-6">
                <div className="text-sm text-neutral-200">
                  Pershendetje <span className="font-semibold">{data.guest.name}</span> 👋
                </div>
                <div className="mt-1 text-sm text-neutral-300">
                  Prek butonin per ta hapur ftesen (muzika fillon).
                </div>

                <button
                  onClick={openInvite}
                  className="mt-5 w-full rounded-xl bg-white px-4 py-3 text-sm font-semibold text-black hover:bg-neutral-200"
                >
                  Hap ftesen
                </button>

                <div className="mt-3 flex items-center justify-between text-xs text-neutral-400">
                  <button
                    onClick={() => setMuted((m) => !m)}
                    className="rounded-lg border border-white/10 px-3 py-1 hover:bg-white/5"
                  >
                    {muted ? "Aktivo muziken" : "Mute"}
                  </button>
                  <div>Scroll poshte per detaje & RSVP</div>
                </div>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Top bar */}
      <div className="sticky top-0 z-40 border-b border-white/10 bg-black/50 backdrop-blur">
        <div className="mx-auto flex max-w-5xl items-center justify-between px-6 py-3">
          <div className="text-sm font-semibold">{inviteTitle}</div>
          <div className="flex items-center gap-2">
            <button
              onClick={() => setMuted((m) => !m)}
              className="rounded-lg border border-white/10 px-3 py-1 text-xs text-neutral-200 hover:bg-white/5"
              aria-label="mute"
            >
              {muted ? "🔈" : "🔇"}
            </button>
            <a
              href="#rsvp"
              className="rounded-lg bg-white/10 px-3 py-1 text-xs text-neutral-100 hover:bg-white/15"
            >
              RSVP
            </a>
          </div>
        </div>
      </div>

      <div className="mx-auto max-w-5xl px-6 py-10">
        {/* Hero */}
        <section className="rounded-3xl border border-white/10 bg-gradient-to-b from-white/10 to-white/5 p-8">
          <motion.div
            initial={{ y: 12, opacity: 0 }}
            animate={{ y: 0, opacity: 1 }}
            transition={{ duration: 0.7 }}
          >
            <div className="text-sm text-neutral-300">Ju jeni te ftuar</div>
            <h1 className="mt-2 text-3xl font-semibold tracking-tight">{inviteTitle}</h1>
            <div className="mt-3 text-neutral-200">
              <span className="text-neutral-400">Data:</span> {fmtDate(data.event.dateISO)}
            </div>
            <div className="mt-2 text-neutral-200">
              <span className="text-neutral-400">Lokacion:</span> {data.event.venueName}
            </div>
            <div className="mt-5 flex flex-wrap items-center gap-3">
              <a
                href={data.event.mapsUrl || "#"}
                target="_blank"
                rel="noreferrer"
                className="rounded-xl bg-white px-4 py-2 text-sm font-semibold text-black hover:bg-neutral-200"
              >
                Hape ne harta
              </a>
              <button
                onClick={() => {
                  const el = document.getElementById("details");
                  el?.scrollIntoView({ behavior: "smooth" });
                }}
                className="rounded-xl border border-white/10 px-4 py-2 text-sm text-neutral-200 hover:bg-white/5"
              >
                Detaje
              </button>
            </div>
          </motion.div>
        </section>

        {/* Details */}
        <section id="details" className="mt-10 grid gap-6 md:grid-cols-2">
          <div className="rounded-2xl border border-white/10 bg-white/5 p-6">
            <div className="text-base font-semibold">Vendndodhja</div>
            <div className="mt-2 text-sm text-neutral-300">{data.event.venueName}</div>
            <div className="mt-1 text-sm text-neutral-400">{data.event.venueAddress}</div>
            {data.event.dressCode ? (
              <div className="mt-4 text-sm text-neutral-300">
                <span className="font-semibold text-neutral-200">Dress code:</span> {data.event.dressCode}
              </div>
            ) : null}
            {data.event.notes ? (
              <div className="mt-3 text-sm text-neutral-300">{data.event.notes}</div>
            ) : null}
          </div>

          <div className="rounded-2xl border border-white/10 bg-white/5 p-6">
            <div className="text-base font-semibold">Itinerari</div>
            <div className="mt-3 space-y-3">
              {(data.event.itinerary || []).length ? (
                (data.event.itinerary || []).map((it, idx) => (
                  <div key={idx} className="flex items-start justify-between gap-3 rounded-xl border border-white/10 bg-black/20 px-4 py-3">
                    <div className="text-sm font-medium">{it.label}</div>
                    <div className="text-sm text-neutral-300">{it.time}</div>
                  </div>
                ))
              ) : (
                <div className="text-sm text-neutral-400">Itinerari ende nuk eshte vendosur.</div>
              )}
            </div>
          </div>
        </section>

        {/* RSVP */}
        <section id="rsvp" className="mt-10 rounded-3xl border border-white/10 bg-white/5 p-6">
          <div className="flex flex-wrap items-center justify-between gap-4">
            <div>
              <div className="text-base font-semibold">Konfirmimi (RSVP)</div>
              <div className="mt-1 text-sm text-neutral-300">
                Ju lutem konfirmoni pjesemarrjen.
              </div>
            </div>
            {data.rsvp ? (
              <div className="rounded-xl border border-white/10 bg-black/20 px-4 py-2 text-sm text-neutral-200">
                Status: <span className="font-semibold">{labelRsvp(data.rsvp.status)}</span> · Persona:{" "}
                <span className="font-semibold">{data.rsvp.count}</span>
              </div>
            ) : null}
          </div>

          <div className="mt-5 grid gap-4 md:grid-cols-3">
            <div className="rounded-2xl border border-white/10 bg-black/20 p-4">
              <div className="text-xs text-neutral-400">Numri i personave</div>
              <input
                id="rsvpCount"
                type="number"
                min={1}
                max={10}
                defaultValue={data.rsvp?.count || 1}
                className="mt-2 w-full rounded-xl border border-white/10 bg-black/40 px-3 py-2 text-sm text-white outline-none focus:border-white/25"
              />
              <div className="mt-4 flex gap-2">
                <button
                  disabled={saving}
                  onClick={() => submitRsvp("yes")}
                  className="flex-1 rounded-xl bg-white px-3 py-2 text-sm font-semibold text-black hover:bg-neutral-200 disabled:opacity-60"
                >
                  Po
                </button>
                <button
                  disabled={saving}
                  onClick={() => submitRsvp("maybe")}
                  className="flex-1 rounded-xl border border-white/10 px-3 py-2 text-sm text-neutral-200 hover:bg-white/5 disabled:opacity-60"
                >
                  Ndoshta
                </button>
                <button
                  disabled={saving}
                  onClick={() => submitRsvp("no")}
                  className="flex-1 rounded-xl border border-white/10 px-3 py-2 text-sm text-neutral-200 hover:bg-white/5 disabled:opacity-60"
                >
                  Jo
                </button>
              </div>
            </div>

            <div className="md:col-span-2 rounded-2xl border border-white/10 bg-black/20 p-4">
              <div className="text-xs text-neutral-400">Mesazh (opsional)</div>
              <textarea
                id="rsvpNote"
                defaultValue={data.rsvp?.note || ""}
                rows={4}
                className="mt-2 w-full resize-none rounded-xl border border-white/10 bg-black/40 px-3 py-2 text-sm text-white outline-none focus:border-white/25"
                placeholder="Nese keni ndonje kerkese ose informate…"
              />
              <div className="mt-3 text-xs text-neutral-500">
                Ruhet automatikisht pasi te klikoni Po / Ndoshta / Jo.
              </div>
            </div>
          </div>
        </section>

        <footer className="mt-12 text-center text-xs text-neutral-500">
          {data.event.coupleA} & {data.event.coupleB} · Ftesa Digjitale
        </footer>
      </div>
    </main>
  );
}

function labelRsvp(s) {
  if (s === "yes") return "Po";
  if (s === "no") return "Jo";
  return "Ndoshta";
}
